#ifdef _WIN32
#include <winsock2.h>
#include <ws2tcpip.h>
#include <stdio.h>
#include <stdlib.h>
#pragma comment(lib, "ws2_32.lib")
#else
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#endif

#define BUF_SIZE 4096

#ifdef _WIN32
#define CLOSESOCK closesocket
#else
#define CLOSESOCK close
#endif

void forward_data(int client_fd, int remote_fd) {
    char buffer[BUF_SIZE];
    fd_set fds;
    int maxfd = (client_fd > remote_fd ? client_fd : remote_fd) + 1;

    while (1) {
        FD_ZERO(&fds);
        FD_SET(client_fd, &fds);
        FD_SET(remote_fd, &fds);

        if (select(maxfd, &fds, NULL, NULL, NULL) < 0) {
            perror("select");
            break;
        }

        if (FD_ISSET(client_fd, &fds)) {
            int n = recv(client_fd, buffer, BUF_SIZE, 0);
            if (n <= 0) break;
            send(remote_fd, buffer, n, 0);
        }

        if (FD_ISSET(remote_fd, &fds)) {
            int n = recv(remote_fd, buffer, BUF_SIZE, 0);
            if (n <= 0) break;
            send(client_fd, buffer, n, 0);
        }
    }
}

int main(int argc, char *argv[]) {
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <port>\n", argv[0]);
        exit(EXIT_FAILURE);
    }

#ifdef _WIN32
    WSADATA wsa;
    if (WSAStartup(MAKEWORD(2,2), &wsa) != 0) {
        fprintf(stderr, "Failed to initialize Winsock\n");
        exit(EXIT_FAILURE);
    }
#endif

    int port = atoi(argv[1]);
    int server_fd = socket(AF_INET, SOCK_STREAM, 0);
    if (server_fd < 0) {
        perror("socket");
        exit(EXIT_FAILURE);
    }

    struct sockaddr_in server_addr;
    memset(&server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    server_addr.sin_port = htons(port);

    if (bind(server_fd, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
        perror("bind");
        exit(EXIT_FAILURE);
    }

    if (listen(server_fd, 5) < 0) {
        perror("listen");
        exit(EXIT_FAILURE);
    }

    printf("SOCKS proxy listening on port %d\n", port);

    while (1) {
        struct sockaddr_in client_addr;
        socklen_t client_len = sizeof(client_addr);
        int client_fd = accept(server_fd, (struct sockaddr *)&client_addr, &client_len);
        if (client_fd < 0) {
            perror("accept");
            continue;
        }

        printf("Connection from %s:%d\n",
               inet_ntoa(client_addr.sin_addr),
               ntohs(client_addr.sin_port));

        // Minimal SOCKS5 handshake
        unsigned char handshake[2];
        recv(client_fd, (char*)handshake, 2, 0);
        unsigned char response[2] = {0x05, 0x00};
        send(client_fd, (char*)response, 2, 0);

        unsigned char req[4];
        recv(client_fd, (char*)req, 4, 0);
        if (req[1] != 0x01) { // only CONNECT supported
            CLOSESOCK(client_fd);
            continue;
        }

        unsigned char atyp;
        recv(client_fd, (char*)&atyp, 1, 0);

        char dest_addr[256];
        int dest_port;
        if (atyp == 0x01) { // IPv4
            unsigned char ip[4];
            recv(client_fd, (char*)ip, 4, 0);
            struct in_addr addr;
            memcpy(&addr, ip, 4);
            strcpy(dest_addr, inet_ntoa(addr));
        } else if (atyp == 0x03) { // Domain name
            unsigned char len;
            recv(client_fd, (char*)&len, 1, 0);
            recv(client_fd, dest_addr, len, 0);
            dest_addr[len] = '\0';
        } else {
            CLOSESOCK(client_fd);
            continue;
        }

        unsigned char port_bytes[2];
        recv(client_fd, (char*)port_bytes, 2, 0);
        dest_port = (port_bytes[0] << 8) | port_bytes[1];

        printf("Forwarding to %s:%d\n", dest_addr, dest_port);

        int remote_fd = socket(AF_INET, SOCK_STREAM, 0);
        struct sockaddr_in remote_addr;
        memset(&remote_addr, 0, sizeof(remote_addr));
        remote_addr.sin_family = AF_INET;
        remote_addr.sin_port = htons(dest_port);
        inet_pton(AF_INET, dest_addr, &remote_addr.sin_addr);

        if (connect(remote_fd, (struct sockaddr *)&remote_addr, sizeof(remote_addr)) < 0) {
            perror("connect");
            CLOSESOCK(client_fd);
            continue;
        }

        unsigned char reply[10] = {0x05, 0x00, 0x00, 0x01};
        memset(reply + 4, 0, 6);
        send(client_fd, (char*)reply, 10, 0);

        forward_data(client_fd, remote_fd);

        CLOSESOCK(client_fd);
        CLOSESOCK(remote_fd);
    }

    CLOSESOCK(server_fd);

#ifdef _WIN32
    WSACleanup();
#endif

    return 0;
}
