#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/select.h>

int main(int argc, char *argv[]) {
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <port>\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    int port = atoi(argv[1]);
    int server_fd, client_fd;
    struct sockaddr_in server_addr, client_addr;
    socklen_t client_len = sizeof(client_addr);

    if ((server_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        perror("socket");
        exit(EXIT_FAILURE);
    }

    int opt = 1;
    setsockopt(server_fd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));

    memset(&server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = INADDR_ANY;
    server_addr.sin_port = htons(port);

    if (bind(server_fd, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
        perror("bind");
        exit(EXIT_FAILURE);
    }

    if (listen(server_fd, 5) < 0) {
        perror("listen");
        exit(EXIT_FAILURE);
    }

    printf("SOCKS proxy listening on port %d...\n", port);

    while (1) {
        client_fd = accept(server_fd, (struct sockaddr *)&client_addr, &client_len);
        if (client_fd < 0) {
            perror("accept");
            continue;
        }

        char client_ip[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, &client_addr.sin_addr, client_ip, sizeof(client_ip));
        printf("New connection from %s:%d\n", client_ip, ntohs(client_addr.sin_port));

        unsigned char buf[262];
        int n = read(client_fd, buf, sizeof(buf));
        if (n <= 0) { close(client_fd); continue; }

        // SOCKS5 greeting: reply with "no authentication"
        unsigned char reply[2] = {0x05, 0x00};
        write(client_fd, reply, 2);

        // Read connection request
        n = read(client_fd, buf, sizeof(buf));
        if (n < 10) { close(client_fd); continue; }

        int cmd = buf[1]; // CONNECT = 0x01
        if (cmd != 0x01) { close(client_fd); continue; }

        struct in_addr dest_ip;
        memcpy(&dest_ip, buf+4, 4);
        int dest_port = (buf[8] << 8) | buf[9];

        char dest_str[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, &dest_ip, dest_str, sizeof(dest_str));
        printf("Forwarding request to %s:%d\n", dest_str, dest_port);

        // Connect to destination
        int remote_fd = socket(AF_INET, SOCK_STREAM, 0);
        struct sockaddr_in dest_addr;
        dest_addr.sin_family = AF_INET;
        dest_addr.sin_addr = dest_ip;
        dest_addr.sin_port = htons(dest_port);

        if (connect(remote_fd, (struct sockaddr*)&dest_addr, sizeof(dest_addr)) < 0) {
            perror("connect");
            close(client_fd);
            continue;
        }

        // Send success reply
        unsigned char success[10] = {0x05, 0x00, 0x00, 0x01};
        memcpy(success+4, &dest_addr.sin_addr, 4);
        memcpy(success+8, &dest_addr.sin_port, 2);
        write(client_fd, success, 10);

        // Relay loop
        fd_set fds;
        char buffer[4096];
        while (1) {
            FD_ZERO(&fds);
            FD_SET(client_fd, &fds);
            FD_SET(remote_fd, &fds);
            int maxfd = (client_fd > remote_fd ? client_fd : remote_fd) + 1;

            if (select(maxfd, &fds, NULL, NULL, NULL) < 0) break;

            if (FD_ISSET(client_fd, &fds)) {
                n = read(client_fd, buffer, sizeof(buffer));
                if (n <= 0) break;
                write(remote_fd, buffer, n);
            }
            if (FD_ISSET(remote_fd, &fds)) {
                n = read(remote_fd, buffer, sizeof(buffer));
                if (n <= 0) break;
                write(client_fd, buffer, n);
            }
        }

        close(remote_fd);
        close(client_fd);
    }

    close(server_fd);
    return 0;
}
