#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>

#define BUF_SIZE 4096

void forward_data(int client_fd, int remote_fd) {
    char buffer[BUF_SIZE];
    fd_set fds;
    int maxfd = (client_fd > remote_fd ? client_fd : remote_fd) + 1;

    while (1) {
        FD_ZERO(&fds);
        FD_SET(client_fd, &fds);
        FD_SET(remote_fd, &fds);

        if (select(maxfd, &fds, NULL, NULL, NULL) < 0) {
            perror("select");
            break;
        }

        if (FD_ISSET(client_fd, &fds)) {
            int n = read(client_fd, buffer, BUF_SIZE);
            if (n <= 0) break;
            write(remote_fd, buffer, n);
        }

        if (FD_ISSET(remote_fd, &fds)) {
            int n = read(remote_fd, buffer, BUF_SIZE);
            if (n <= 0) break;
            write(client_fd, buffer, n);
        }
    }
}

int main(int argc, char *argv[]) {
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <port>\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    int port = atoi(argv[1]);
    int server_fd = socket(AF_INET, SOCK_STREAM, 0);
    if (server_fd < 0) {
        perror("socket");
        exit(EXIT_FAILURE);
    }

    struct sockaddr_in server_addr = {0};
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = INADDR_ANY;
    server_addr.sin_port = htons(port);

    if (bind(server_fd, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
        perror("bind");
        exit(EXIT_FAILURE);
    }

    if (listen(server_fd, 5) < 0) {
        perror("listen");
        exit(EXIT_FAILURE);
    }

    printf("SOCKS proxy listening on port %d\n", port);

    while (1) {
        struct sockaddr_in client_addr;
        socklen_t client_len = sizeof(client_addr);
        int client_fd = accept(server_fd, (struct sockaddr *)&client_addr, &client_len);
        if (client_fd < 0) {
            perror("accept");
            continue;
        }

        printf("Connection from %s:%d\n",
               inet_ntoa(client_addr.sin_addr),
               ntohs(client_addr.sin_port));

        // SOCKS5 handshake (no authentication)
        unsigned char handshake[2];
        read(client_fd, handshake, 2);
        unsigned char response[2] = {0x05, 0x00};
        write(client_fd, response, 2);

        // SOCKS5 request
        unsigned char req[4];
        read(client_fd, req, 4);
        if (req[1] != 0x01) { // only CONNECT supported
            close(client_fd);
            continue;
        }

        unsigned char atyp;
        read(client_fd, &atyp, 1);

        char dest_addr[256];
        int dest_port;
        if (atyp == 0x01) { // IPv4
            unsigned char ip[4];
            read(client_fd, ip, 4);
            struct in_addr addr;
            memcpy(&addr, ip, 4);
            strcpy(dest_addr, inet_ntoa(addr));
        } else if (atyp == 0x03) { // Domain name
            unsigned char len;
            read(client_fd, &len, 1);
            read(client_fd, dest_addr, len);
            dest_addr[len] = '\0';
        } else {
            close(client_fd);
            continue;
        }

        unsigned char port_bytes[2];
        read(client_fd, port_bytes, 2);
        dest_port = (port_bytes[0] << 8) | port_bytes[1];

        printf("Forwarding to %s:%d\n", dest_addr, dest_port);

        // Connect to remote
        int remote_fd = socket(AF_INET, SOCK_STREAM, 0);
        struct sockaddr_in remote_addr = {0};
        remote_addr.sin_family = AF_INET;
        remote_addr.sin_port = htons(dest_port);
        inet_pton(AF_INET, dest_addr, &remote_addr.sin_addr);

        if (connect(remote_fd, (struct sockaddr *)&remote_addr, sizeof(remote_addr)) < 0) {
            perror("connect");
            close(client_fd);
            continue;
        }

        // Send success reply
        unsigned char reply[10] = {0x05, 0x00, 0x00, 0x01};
        memset(reply + 4, 0, 6);
        write(client_fd, reply, 10);

        forward_data(client_fd, remote_fd);

        close(client_fd);
        close(remote_fd);
    }

    close(server_fd);
    return 0;
}
