#include "PingWidget.h"
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>

PingWidget::PingWidget(QWidget *parent) : QWidget(parent) {
    hostInput = new QLineEdit(this);
    hostInput->setPlaceholderText("Enter host (e.g., google.com)");

    pingButton = new QPushButton("Ping", this);
    outputArea = new QTextEdit(this);
    outputArea->setReadOnly(true);

    pingProcess = new QProcess(this);
    connect(pingButton, &QPushButton::clicked, this, &PingWidget::startPing);
    connect(pingProcess, &QProcess::readyReadStandardOutput, this, &PingWidget::readPingOutput);
    connect(pingProcess, QOverload<int, QProcess::ExitStatus>::of(&QProcess::finished),
            this, &PingWidget::handlePingFinished);

    auto *layout = new QVBoxLayout(this);
    auto *topLayout = new QHBoxLayout();
    topLayout->addWidget(new QLabel("Host:"));
    topLayout->addWidget(hostInput);
    topLayout->addWidget(pingButton);

    layout->addLayout(topLayout);
    layout->addWidget(outputArea);
    setLayout(layout);
    setWindowTitle("Qt Ping Utility");
    resize(600, 400);
}

void PingWidget::startPing() {
    outputArea->clear();
    QString host = hostInput->text().trimmed();
    if (host.isEmpty()) {
        outputArea->setText("Please enter a valid host.");
        return;
    }

#ifdef Q_OS_WIN
    QStringList args = {host};
    pingProcess->start("ping", args);
#else
    QStringList args = {"-c", "4", host};  // Send 4 packets
    pingProcess->start("ping", args);
#endif
}

void PingWidget::readPingOutput() {
    QString output = pingProcess->readAllStandardOutput();
    outputArea->append(output);
}

void PingWidget::handlePingFinished(int exitCode, QProcess::ExitStatus status) {
    if (status == QProcess::CrashExit) {
        outputArea->append("\nPing process crashed.");
    } else {
        outputArea->append("\nPing finished.");
    }
}
